#!/bin/sh

# kernel 書き込み 最大リトライ回数
MAX_RETRY_COUNT=5

DIR=`dirname "${0}"`
DIR_FULL_PATH="${PWD}/${DIR}"

KERNEL_FILE=$DIR/nandflash/mtd1/kernel
KERNEL_SPLIT_FILE_PREFIX=$DIR/nandflash/mtd1/kernel_split.

# for loop
MTD_SIZE=`cat /sys/class/mtd/mtd1/size`
MTD_ERASE_SIZE=`cat /sys/class/mtd/mtd1/erasesize`
KERNEL_SIZE=8388608
WRITE_LOOP_MAX_COUNT=`expr ${MTD_SIZE} \/ ${MTD_ERASE_SIZE}`
WRITE_LOOP_COUNT=`expr ${KERNEL_SIZE} \/ ${MTD_ERASE_SIZE}`
WRITE_LOOP_COUNT_MOD=`expr ${KERNEL_SIZE} \% ${MTD_ERASE_SIZE}`

if [ ${WRITE_LOOP_COUNT_MOD} -ne 0 ];then
  WRITE_LOOP_COUNT=`expr ${WRITE_LOOP_COUNT} \+ 1`
fi

SUFFIX_ARRAY=(`seq -w 0 $(expr ${WRITE_LOOP_MAX_COUNT} \- 1)`)

versionup_kernel() {
  # カーネルファイル分割
  # Eraseサイズでカーネルファイルを分割する
  rm -f ${KERNEL_SPLIT_FILE_PREFIX}*
  split -d -b ${MTD_ERASE_SIZE} ${KERNEL_FILE} ${KERNEL_SPLIT_FILE_PREFIX}

  WRITE_COUNT=0 

  # 最大書き込み回数分ループする
  for i in `seq -w 0 $(expr ${WRITE_LOOP_MAX_COUNT} \- 1)` ; do
    #
    # create filename
    #
    KERNEL_SPLIT_FILE=${KERNEL_SPLIT_FILE_PREFIX}${SUFFIX_ARRAY[${WRITE_COUNT}]}
    KERNEL_SPLIT_FILE_READCHECK=${KERNEL_SPLIT_FILE_PREFIX}${SUFFIX_ARRAY[${WRITE_COUNT}]}.readcheck

    #
    # kernel 書き込み処理
    #
    WRITE_OFFSET=`expr ${i} \* ${MTD_ERASE_SIZE}`
    #mtd_debug erase /dev/mtd1 0 0x1000000
    mtd_debug erase /dev/mtd1 ${WRITE_OFFSET} ${MTD_ERASE_SIZE}

    if [ $? -ne 0 ];then
      # bad block skip
      echo write kernel skip. erase error. offset: ${WRITE_OFFSET}
      continue
    fi

    #mtd_debug write /dev/mtd1 0 0x1000000 ${KERNEL_FILE}
    mtd_debug write /dev/mtd1 ${WRITE_OFFSET} ${MTD_ERASE_SIZE} ${KERNEL_SPLIT_FILE}

    #
    # kernel 読み込み処理
    #
    #mtd_debug read /dev/mtd1 0 0x1000000 ${KERNEL_FILE}.readcheck
    mtd_debug read /dev/mtd1 ${WRITE_OFFSET} ${MTD_ERASE_SIZE} ${KERNEL_SPLIT_FILE_READCHECK}

    #
    # kernel 比較処理
    #
    cmp ${KERNEL_SPLIT_FILE} ${KERNEL_SPLIT_FILE_READCHECK}
    if [ $? -eq 0 ];then
      # 書き込みデータと読み込みデータが一致した場合
      # 書き込みが正常に行われたとみなす
      echo write kernel success. ${KERNEL_SPLIT_FILE}
      WRITE_COUNT=`expr ${WRITE_COUNT} \+ 1`
    else
      echo write kernel skip. compare error. offset: ${WRITE_OFFSET}, file: ${KERNEL_SPLIT_FILE}
      continue
    fi

    #
    # kernel 書き込み終了条件チェック
    #
    TOTAL_WRITE_SIZE=`expr ${WRITE_COUNT} \* ${MTD_ERASE_SIZE}`
    if [ ${TOTAL_WRITE_SIZE} -ge ${KERNEL_SIZE} ];then
      # sync filesystem
      sync
      # 正常終了
      exit 0
    fi
  done
}

versionup_kernel_retry_loop() {
  for i in `seq 1 ${MAX_RETRY_COUNT}` ; do
    echo write count: $i
    # version up kernel
    versionup_kernel
    echo write kernel failed.
  done
}

if [ -e /dev/mtd1 ]; then
  echo exist /dev/mtd1

  if [ -e ${KERNEL_FILE} ]; then
    echo exist ${KERNEL_FILE}
    # nand flash版の場合のみカーネルの更新処理を行う
    # versin up kernel retry loop
    versionup_kernel_retry_loop
  fi
fi

# ここまで到達した場合は書き込み失敗
exit -1
